<?php
/**
 * Order Status Rules for WooCommerce - Meta Boxes Class
 *
 * @version 2.4.0
 * @since   1.4.0
 *
 * @author  Algoritmika Ltd.
 */

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! class_exists( 'Alg_WC_Order_Status_Rules_Meta_Boxes' ) ) :

class Alg_WC_Order_Status_Rules_Meta_Boxes {

	/**
	 * Constructor.
	 *
	 * @version 1.4.0
	 * @since   1.4.0
	 */
	function __construct() {
		add_action( 'add_meta_boxes', array( $this, 'add_status_change_meta_box' ) );
	}

	/**
	 * get_core.
	 *
	 * @version 1.6.0
	 * @since   1.6.0
	 */
	function get_core() {
		if ( ! isset( $this->core ) ) {
			$this->core = alg_wc_order_status_rules()->core;
		}
		return $this->core;
	}

	/**
	 * add_status_change_meta_box.
	 *
	 * @version 1.0.0
	 * @since   1.0.0
	 */
	function add_status_change_meta_box() {
		add_meta_box(
			'alg-wc-order-status-change-history',
			__( 'Order Status History', 'order-status-rules-for-woocommerce' ),
			array( $this, 'create_status_change_meta_box' ),
			'shop_order',
			'normal',
			'default'
		);
	}

	/**
	 * create_status_change_meta_box.
	 *
	 * @version 2.4.0
	 * @since   1.0.0
	 *
	 * @todo    [later] (desc) `( ! $is_status_match )`: better desc
	 */
	function create_status_change_meta_box() {
		$order_id       = get_the_ID();
		$status_history = $this->get_core()->get_order_status_change_history( $order_id );
		if ( empty( $status_history ) ) {
			echo '<p><em>' . __( 'No data.', 'order-status-rules-for-woocommerce' ) . '</p></em>';
		} else {
			// Status history
			$date_format    = $this->get_core()->get_date_time_format();
			$status_history = array_reverse( $status_history, true );
			$last_record    = current( $status_history );
			$status         = wc_get_order_statuses();
			echo '<table class="widefat stripped">' .
				'<tr>' .
					'<th>' . __( 'Nr.', 'order-status-rules-for-woocommerce' )  . '</th>' .
					'<th>' . __( 'Time', 'order-status-rules-for-woocommerce' ) . '</th>' .
					'<th>' . __( 'From', 'order-status-rules-for-woocommerce' ) . '</th>' .
					'<th>' . __( 'To', 'order-status-rules-for-woocommerce' )   . '</th>' .
				'</tr>';
			foreach ( $status_history as $index => $record ) {
				$i    = ( $index + 1 );
				$time = date_i18n( $date_format, $record['time'] );
				$from = '<code>' . ( isset( $status[ 'wc-' . $record['from'] ] ) ? $status[ 'wc-' . $record['from'] ] : $record['from'] ) . '</code>';
				$to   = '<code>' . ( isset( $status[ 'wc-' . $record['to'] ] )   ? $status[ 'wc-' . $record['to'] ]   : $record['to'] )   . '</code>';
				echo "<tr><td>{$i}</td><td>{$time}</td><td>{$from}</td><td>{$to}</td></tr>";
			}
			echo '</table>';
			// Scheduled status update
			$this->get_core()->init_options();
			$order               = wc_get_order( $order_id );
			$args                = $this->get_core()->get_order_data( $order );
			$is_status_match     = ( $last_record['to'] === $args['order_status'] );
			$last_record['to']   = ( ! $is_status_match && $this->get_core()->do_use_last_record ? $args['order_status'] : $last_record['to'] );
			$args['last_record'] = $last_record;
			$is_rule_applied     = false;
			foreach ( $this->get_core()->options['from'] as $i => $from ) {
				$args['from'] = $from;
				if ( $this->get_core()->do_apply_rule( $i, $args ) ) {
					$unit           = ( isset( $this->get_core()->options['time_trigger_units'][ $i ] ) ? $this->get_core()->options['time_trigger_units'][ $i ] : 'hour' );
					$step           = $this->get_core()->get_trigger_unit_step( $unit );
					$skip           = ( isset( $this->get_core()->options['skip_days'][ $i ] ) ? $this->get_core()->options['skip_days'][ $i ] : false );
					$current_time   = current_time( 'timestamp' );
					$time_remaining = $this->get_core()->get_time_remaining( $last_record['time'], $this->get_core()->options['time_triggers'][ $i ] * $step, $skip, $current_time );
					$to             = $this->get_core()->options['to'][ $i ];
					$rule           = sprintf( __( 'Rule #%s', 'order-status-rules-for-woocommerce' ), $i ) .
						( ! empty( $this->get_core()->options['titles'][ $i ] ) ? ': ' . $this->get_core()->options['titles'][ $i ] : '' );
					echo '<p><em>' .
						sprintf( __( 'Status scheduled to be updated from %s to %s (%s) on %s (i.e. %s).', 'order-status-rules-for-woocommerce' ),
							'<code>' . ( isset( $status[ $from ] ) ? $status[ $from ] : $from ) . '</code>',
							'<code>' . ( isset( $status[ $to ] )   ? $status[ $to ]   : $to )   . '</code>',
							$rule,
							date_i18n( $date_format, $current_time + $time_remaining ),
							( $time_remaining > 0 ?
								sprintf( __( 'in %s', 'order-status-rules-for-woocommerce' ), human_time_diff( $current_time - $time_remaining, $current_time ) ) :
								__( 'now', 'order-status-rules-for-woocommerce' ) )
						) . ' ' .
						sprintf( __( 'Current time is %s.', 'order-status-rules-for-woocommerce' ),
							date_i18n( $date_format, $current_time ) ) .
					'</p></em>';
					$is_rule_applied = true;
					break;
				}
			}
			if ( ! $is_rule_applied ) {
				echo '<p><em>' . __( 'No order status rules are scheduled to be applied for the current order.', 'order-status-rules-for-woocommerce' ) . '</em></p>';
			}
			// Check matching order status
			if ( ! $is_status_match ) {
				echo '<p>';
				if ( $this->get_core()->do_use_last_record ) {
					echo '<span class="dashicons dashicons-info"></span> ' .
						__( 'Although the current order status does not match the last record in the order history, it will be used anyway.', 'order-status-rules-for-woocommerce' );
				} else {
					echo '<span class="dashicons dashicons-warning" style="color:red;"></span> ' .
						__( 'The current order status does not match the last record in the order history! Order status rules will not be applied!', 'order-status-rules-for-woocommerce' );
				}
				echo ' ' . sprintf( __( 'To change this behaviour, please check the "%s" option in the %s section.', 'order-status-rules-for-woocommerce' ),
					__( 'On non-matching order status', 'order-status-rules-for-woocommerce' ),
					'<a href="' . admin_url( 'admin.php?page=wc-settings&tab=alg_wc_order_status_rules&section=advanced' ) . '" target="_blank">' .
						__( 'Advanced', 'order-status-rules-for-woocommerce' ) . '</a>' );
				echo '</p>';
			}
		}
	}

}

endif;

return new Alg_WC_Order_Status_Rules_Meta_Boxes();
