<?php namespace FBLiveChat;

use WP_Query;

/**
 * Settings Page
 *
 * @since 1.0.0
 * @package FBLiveChat
 */
final class SettingsPage
{
    /**
     * @var string
     */
    const SLUG = 'facebook-messenger-livechat-settings';

    /**
     * @var string
     */
    const GROUP = 'fblivechat_settings_group';

    /**
     * Current settings
     *
     * @var array
     */
    private $settings;

    /**
     * @var string
     */
    public $hook_suffix;

    /**
     * Constructor
     */
    function __construct()
    {
        $this->settings = fblivechat_get_settings();

        add_action('admin_menu', [$this, '_adminMenu']);
        add_action('admin_init', [$this, '_adminInit'], 10, 0);
        add_action('admin_notices', [$this, '_adminNotices'], 10, 0);
        add_action('admin_enqueue_scripts', [$this, '_adminEnqueueScripts']);
    }

    /**
     * @internal Used as a callback.
     */
    function _adminInit()
    {
        register_setting(self::GROUP, Plugin::OPTION_KEY, [$this, '_sanitize']);
    }

    /**
     * @internal Used as a callback.
     */
    function _sanitize(array $settings)
    {
        if (isset($settings['icon_offset_left']) && '' !== $settings['icon_offset_left']) {
            $settings['icon_offset_left'] = intval($settings['icon_offset_left']);
        }

        if (isset($settings['icon_offset_right']) && '' !== $settings['icon_offset_right']) {
            $settings['icon_offset_right'] = intval($settings['icon_offset_right']);
        }

        if (isset($settings['icon_offset_bottom']) && '' !== $settings['icon_offset_bottom']) {
            $settings['icon_offset_bottom'] = intval($settings['icon_offset_bottom']);
        }

        if (empty($settings['chat_enabled'])) {
            $settings['chat_enabled'] = 0;
        }

        return $settings;
    }

    /**
     * @internal Used as a callback
     */
    function _adminMenu($context)
    {
        $this->hook_suffix = add_menu_page(
            esc_html__('Facebook Messenger Live Chat Settings', 'facebook-messenger-live-chat-for-wordpress'),
            esc_html__('Messenger Chat', 'facebook-messenger-live-chat-for-wordpress'),
            'manage_options',
            self::SLUG,
            [$this, '_render'],
            FBMLC_URI . 'assets/icons/messenger.svg'
        );
    }

    /**
     * @internal
     */
    function _adminNotices()
    {
        if ($GLOBALS['page_hook'] !== $this->hook_suffix) {
            return;
        }

        if (isset($_REQUEST['settings-updated']) && 'true' === $_REQUEST['settings-updated']) {
            echo '<div class="updated notice is-dismissible"><p><strong>' . esc_html__('Settings have been saved successfully!', 'facebook-messenger-live-chat-for-wordpress') . '</strong></p><button type="button" class="notice-dismiss"><span class="screen-reader-text">' . esc_html__('Dismiss this notice.', 'facebook-messenger-live-chat-for-wordpress') . '</span></div>';
        }

        if (isset($_REQUEST['error']) && 'true' === $_REQUEST['error']) {
            echo '<div class="updated error is-dismissible"><p><strong>' . esc_html__('Failed to save settings. Please try again!', 'facebook-messenger-live-chat-for-wordpress') . '</strong></p><button type="button" class="notice-dismiss"><span class="screen-reader-text">' . esc_html__('Dismiss this notice.', 'facebook-messenger-live-chat-for-wordpress') . '</span></div>';
        }
    }

    /**
     * @internal
     */
    function _adminEnqueueScripts($hook_suffix)
    {
        if ($hook_suffix !== $this->hook_suffix) {
            return;
        }

        wp_enqueue_style('select2', 'https://cdn.jsdelivr.net/npm/select2@latest/dist/css/select2.min.css', [], 'latest');
        wp_enqueue_script('select2', 'https://cdn.jsdelivr.net/npm/select2@latest/dist/js/select2.min.js', ['jquery-core'], 'latest', true);

        wp_enqueue_style('settings-page', FBMLC_URI . 'assets/css/settings-page.min.css', ['wp-color-picker'], Plugin::VERSION);
        wp_enqueue_script('settings-page', FBMLC_URI . 'assets/js/settings-page.min.js', ['wp-color-picker'], Plugin::VERSION, true);

        wp_localize_script('settings-page', 'FBLiveChat', [
            'homeUrl' => home_url(),
            'textDirection' => is_rtl() ? 'rtl' : 'ltr',
            'clientLocale' => !empty($this->settings['client_locale']) ? $this->settings['client_locale'] : get_locale()
        ]);
    }

    /**
     * Render
     */
    function _render($data)
    {
        $status = -1;
        $chat_api = new MessengerChatApi();
        $app_id = $this->value('app_id', 0);
        $app_secret = $this->value('app_secret', 0);
        $client_token = $this->value('client_token', 0);

        if ($client_token && $app_id && $app_secret) {
            $status = $chat_api->getTokenStatus($client_token);
            if (gettype($status) == 'string') {
                $status = -1;
            }
        }
        ?>
        <div class="wrap fblivechat-settings-page-container">
            <h1 class="admin-page-title"><?php esc_html_e('Clever Facebook Messenger Live Chat Settings', 'facebook-messenger-live-chat-for-wordpress') ?></h1>
            <form action="options.php" method="post">
                <?php settings_fields(self::GROUP); ?>
                <ul class="fblivechat-tabs-control">
                    <li class="fblivechat-tab-control-item active">
                        <a href="#fblivechat-general" title="<?php esc_attr_e('General', 'facebook-messenger-live-chat-for-wordpress') ?>"><?php esc_html_e('General', 'facebook-messenger-live-chat-for-wordpress') ?></a>
                    </li>
                    <li class="fblivechat-tab-control-item">
                        <a href="#fblivechat-chatbox" title="<?php esc_attr_e('Chatbox', 'facebook-messenger-live-chat-for-wordpress') ?>"><?php esc_html_e('Chatbox', 'facebook-messenger-live-chat-for-wordpress') ?></a>
                    </li>
                </ul>
                <div class="fblivechat-tabs-content">
                    <div id="fblivechat-general" class="fblivechat-tab active">
                        <div class="fblivechat-tab-content">
                            <table class="form-table">
                                <tr>
                                    <th scope="row"><?php esc_html_e('Enable Live Chat', 'facebook-messenger-live-chat-for-wordpress') ?></th>
                                    <td>
                                        <label>
                                            <input id="fblivechat-enabled" type="checkbox"
                                                   name="<?php $this->field('chat_enabled') ?>"
                                                   value="1"<?php checked($this->value('chat_enabled', 0), 1) ?>>
                                            <span class="description"><?php esc_html_e('Will be enabled if checked.', 'facebook-messenger-live-chat-for-wordpress') ?></span>
                                        </label>
                                    </td>
                                </tr>
                            </table>
                            <!-- <h3 class="fblivechat-heading-block"><?php esc_html_e('Account Settings', 'facebook-messenger-live-chat-for-wordpress') ?></h3> -->
                            <table class="form-table">
                                <?php if ($status != 1) { ?>
                                    <tr>
                                        <th scope="row"><?php esc_html_e('Connect to Facebook', 'facebook-messenger-live-chat-for-wordpress') ?></th>
                                        <td>
                                            <a href="javascript:void(0)"
                                               class="button button-primary fblivechat-button-connect"><?php esc_html_e('Connect now', 'facebook-messenger-live-chat-for-wordpress') ?></a>
                                            <p class="description"><?php esc_html_e('Log into your Facebook account to setup Messenger on this site.', 'facebook-messenger-live-chat-for-wordpress') ?></p>
                                        </td>
                                    </tr>
                                <?php }
                                if ($status == 1) {
                                    $client = $chat_api->getUser($client_token);
                                    $client_pages = $chat_api->getUserPages($client_token);
                                    ?>
                                    <tr>
                                        <th scope="row">
                                            <label for=""><?php esc_html_e('Linked Account', 'facebook-messenger-live-chat-for-wordpress') ?></label>
                                        </th>
                                        <td>
                                            <p><img src="<?php echo $client->getPicture()->getUrl() ?>"></p>
                                            <p><?php esc_html_e('Logged in as', 'facebook-messenger-live-chat-for-wordpress') ?>
                                                <b><?php echo $client->getName(); ?>.</b>
                                                <a href="javascript:void(0)"
                                                   class="fblivechat-button-reconnect"><?php esc_html_e('Change?', 'facebook-messenger-live-chat-for-wordpress') ?></a>
                                            </p>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row"><?php esc_html_e('Facebook Page', 'facebook-messenger-live-chat-for-wordpress') ?></label></th>
                                        <td>
                                            <?php $client_page_id = $this->value('client_page_id', 0); ?>
                                            <select name="<?php $this->field('client_page_id') ?>">
                                                <option value=""><?php esc_html_e('Select a Page', 'facebook-messenger-live-chat-for-wordpress') ?></option>
                                                <?php foreach ($client_pages['accounts'] as $key => $page) : ?>
                                                    <option value="<?php echo $page['id']; ?>"<?php selected($page['id'], $client_page_id) ?>><?php echo $page['name']; ?></option>
                                                <?php endforeach; ?>
                                            </select>
                                            <p class="description"><?php esc_html_e('Select a page which you want your visitors to interact with.', 'facebook-messenger-live-chat-for-wordpress') ?></p>
                                        </td>
                                    </tr>
                                <?php } ?>
                                <tr>
                                    <th scope="row"><?php esc_html_e('Language', 'facebook-messenger-live-chat-for-wordpress') ?></th>
                                    <td>
                                        <?php
                                        $locales = fblivechat_get_locales();
                                        $client_locale = $this->value('client_locale', 0) ?: get_locale();
                                        ?>
                                        <select name="<?php $this->field('client_locale') ?>">
                                            <?php foreach ($locales as $key => $value) { ?>
                                                <option value="<?php echo $key; ?>"<?php selected($client_locale, $key) ?>><?php echo $value; ?></option>
                                            <?php } ?>
                                        </select>
                                    </td>
                                </tr>
                            </table>
                        </div>
                    </div>
                    <div id="fblivechat-chatbox" class="fblivechat-tab">
                        <div class="fblivechat-tab-content">
                            <!-- <h3 class="fblivechat-heading-block"><?php esc_html_e('Chatbox Settings', 'facebook-messenger-live-chat-for-wordpress') ?></h3> -->
                            <table class="form-table">
                                <tr>
                                    <th scope="row"><?php esc_html_e('Show on Device(s)', 'facebook-messenger-live-chat-for-wordpress') ?></th>
                                    <td>
                                        <?php $show_on_device = $this->value('show_on_device', 0) ?>
                                        <select name="<?php $this->field('show_on_device') ?>"
                                                id="fblivechat-show-on-device">
                                            <option value="mobile"<?php selected('mobile', $show_on_device) ?>><?php _e('Mobile', 'facebook-messenger-live-chat-for-wordpress'); ?></option>
                                            <option value="desktop"<?php selected('desktop', $show_on_device) ?>><?php _e('Desktop', 'facebook-messenger-live-chat-for-wordpress'); ?></option>
                                            <option value="both"<?php selected('both', $show_on_device) ?>><?php _e('Desktop & Mobile', 'facebook-messenger-live-chat-for-wordpress'); ?></option>
                                        </select>
                                        <p class="description"><?php esc_html_e('Which device(s) do you want to display Messenger chatbox?', 'facebook-messenger-live-chat-for-wordpress') ?></p>
                                    </td>
                                </tr>
                                <tr id="fblivechat-minimize-on-desktop-row">
                                    <th scope="row"><?php esc_html_e('Minimize on Desktop', 'facebook-messenger-live-chat-for-wordpress') ?></th>
                                    <td>
                                        <label>
                                            <input type="checkbox" name="<?php $this->field('minimize_on_desktop') ?>"<?php checked($this->value('minimize_on_desktop', 0), 1) ?> value="1">
                                            <span class="description"><?php esc_html_e('By default, chatbox will be minimized on mobile only. To minimize on desktop as well, tick this option.', 'facebook-messenger-live-chat-for-wordpress') ?></span>
                                        </label>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php echo __('Show on Page(s)', 'facebook-messenger-live-chat-for-wordpress') ?></th>
                                    <td>
                                        <?php $show_on_page = $this->value('show_on_page', 0) ?>
                                        <select name="<?php $this->field('show_on_page') ?>" id="fblivechat-show-on">
                                            <option value="all"<?php selected($show_on_page, 'all') ?>><?php echo __('All pages and posts', 'facebook-messenger-live-chat-for-wordpress') ?></option>
                                            <option value="custom1"<?php selected($show_on_page, 'custom1') ?>><?php echo __('All pages but except...', 'facebook-messenger-live-chat-for-wordpress') ?></option>
                                            <option value="custom2"<?php selected($show_on_page, 'custom2') ?>><?php echo __('Display on pages...', 'facebook-messenger-live-chat-for-wordpress') ?></option>
                                        </select>
                                    </td>
                                </tr>
                                <tr id="fblivechat-excluded-pages-row">
                                    <th scope="row"><?php echo __('Hide on pages', 'facebook-messenger-live-chat-for-wordpress') ?></th>
                                    <td>
                                        <?php
                                        $posts = fblivechat_list_pages();
                                        $selected = (array)$this->value('hide_pages', 0);
                                        ?>
                                        <?php if (!empty($posts)) { ?>
                                            <select class="select2" name="<?php $this->field('hide_pages') ?>[]" multiple>
                                                <?php foreach ($posts as $slug => $title) { ?>
                                                    <option value="<?php esc_attr_e($slug) ?>"<?php $this->multiSelected($slug, $selected) ?>><?php esc_html_e($title) ?></option>
                                                <?php } ?>
                                            </select>
                                        <?php } ?>
                                        <p class="description"><?php _e('Select the page(s) which you DO NOT want to show Messenger Chatbox.', 'facebook-messenger-live-chat-for-wordpress') ?></p>
                                    </td>
                                </tr>
                                <tr id="fblivechat-included-pages-row">
                                    <th scope="row"><?php echo __('Show on pages', 'facebook-messenger-live-chat-for-wordpress') ?></th>
                                    <td>
                                        <?php
                                        $posts = fblivechat_list_pages();
                                        $selected = (array)$this->value('show_pages', 0);
                                        ?>
                                        <?php if (!empty($posts)) { ?>
                                            <select class="select2" name="<?php $this->field('show_pages') ?>[]" multiple>
                                                <?php foreach ($posts as $slug => $title) { ?>
                                                    <option value="<?php esc_attr_e($slug) ?>"<?php $this->multiSelected($slug, $selected) ?>><?php esc_html_e($title) ?></option>
                                                <?php } ?>
                                            </select>
                                        <?php } ?>
                                        <p class="description"><?php _e('Select the page(s) which you want to show Messenger Chatbox.', 'facebook-messenger-live-chat-for-wordpress') ?></p>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php esc_html_e('Icon Offset', 'facebook-messenger-live-chat-for-wordpress') ?></th>
                                    <td>
                                        <div class="flex-container">
                                            <label class="inline-box">
                                                <span><?php esc_html_e('Left (px)', 'facebook-messenger-live-chat-for-wordpress') ?></span>
                                                <input type="text" name="<?php $this->field('icon_offset_left') ?>"
                                                       value="<?php $this->value('icon_offset_left') ?>">
                                            </label>
                                            <label class="inline-box">
                                                <span><?php esc_html_e('Right (px)', 'facebook-messenger-live-chat-for-wordpress') ?></span>
                                                <input type="text" name="<?php $this->field('icon_offset_right') ?>"
                                                       value="<?php $this->value('icon_offset_right') ?>">
                                            </label>
                                            <label class="inline-box">
                                                <span><?php esc_html_e('Bottom (px)', 'facebook-messenger-live-chat-for-wordpress') ?></span>
                                                <input type="text" name="<?php $this->field('icon_offset_bottom') ?>"
                                                       value="<?php $this->value('icon_offset_bottom') ?>">
                                            </label>
                                        </div>
                                        <p class="description"><?php esc_html_e('Leave empty to unset an offset.', 'facebook-messenger-live-chat-for-wordpress') ?></p>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php esc_html_e('Theme Color') ?></th>
                                    <td>
                                        <input class="wp-color-picker" type="text"
                                               name="<?php $this->field('theme_color') ?>"
                                               value="<?php $this->value('theme_color'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php esc_html_e('Logged in Greeting', 'facebook-messenger-live-chat-for-wordpress') ?></th>
                                    <td>
                                <textarea name="<?php $this->field('logged_in_greeting') ?>" rows="3"
                                          cols="50"><?php $this->value('logged_in_greeting') ?></textarea>
                                        <p class="description">
                                            <?php _e('The greeting message that will be displayed if the user is currently logged in to Facebook. Maximum 80 characters.', 'facebook-messenger-live-chat-for-wordpress'); ?>
                                        </p>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php esc_html_e('Logged out Greeting', 'facebook-messenger-live-chat-for-wordpress') ?></th>
                                    <td>
                                <textarea name="<?php $this->field('logged_out_greeting') ?>" rows="3"
                                          cols="50"><?php $this->value('logged_out_greeting') ?></textarea>
                                        <p class="description">
                                            <?php _e('The greeting message that will be displayed if the user is currently not logged in to Facebook. Maximum 80 characters.', 'facebook-messenger-live-chat-for-wordpress'); ?>
                                        </p>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php esc_html_e('Show Greeting Dialog After', 'facebook-messenger-live-chat-for-wordpress') ?></th>
                                    <td>
                                        <input type="number" name="<?php $this->field('greeting_dialog_delay') ?>"
                                               value="<?php $this->value('greeting_dialog_delay') ?>"> <?php esc_html_e('seconds', 'facebook-messenger-live-chat-for-wordpress') ?>
                                        <p class="description"><?php esc_html_e('Sets the number of seconds of delay before the greeting dialog is shown after your site is fully loaded.', 'facebook-messenger-live-chat-for-wordpress') ?></p>
                                    </td>
                                </tr>
                            </table>

                        </div>
                    </div>
                    <?php submit_button() ?>
                    <?php if ($status == 1) { ?>
                        <input type="hidden" name="<?php $this->field('app_id') ?>"
                               value="<?php $this->value('app_id') ?>">
                        <input type="hidden" name="<?php $this->field('app_secret') ?>"
                               value="<?php $this->value('app_secret') ?>">
                        <input type="hidden" name="<?php $this->field('client_token') ?>"
                               value="<?php $this->value('client_token') ?>">
                    <?php } ?>
                </div>
            </form>
        </div>
        <?php
    }

    /**
     * Get a field key
     */
    private function field($name, $echo = true)
    {
        $field = Plugin::OPTION_KEY . '[' . $name . ']';

        if ($echo) {
            echo $field;
        } else {
            return $field;
        }
    }

    /**
     * Get a field value
     */
    private function value($name, $echo = true)
    {
        $value = isset($this->settings[$name]) ? $this->settings[$name] : '';

        if ($echo) {
            echo $value;
        } else {
            return $value;
        }
    }

    /**
     * Multiple selected
     *
     * @return bool
     */
    private function multiSelected($test, array $values, $echo = true)
    {
        $test = strval($test);
        $values = array_map('strval', $values);
        $selected = in_array($test, $values) ? ' selected' : '';

        if ($echo) {
            echo $selected;
        } else {
            return $selected;
        }
    }
}

if (is_admin()) {
    return new SettingsPage();
}
