<?php namespace FBLiveChat;

use Exception;

/**
 * Plugin Name: Facebook Messenger Live Chat for WordPress
 * Description: Bring Messenger chatbox into your website to increase your customer engagement.
 * Plugin URI: https://cleveraddon.com/clever-facebook-messenger-live-chat/
 * Author: CleverAddon
 * Version: 1.0.0
 * Author URI: https://cleveraddon.com/
 */
final class Plugin
{
    /**
     * @var string
     */
    const VERSION = '1.0.0';

    /**
     * @var string
     */
    const OPTION_KEY = 'fb_messenger_live_chat_options';

    /**
     * Constructor
     */
    public function __construct()
    {
        add_action('plugins_loaded', [$this, '_install'], 10, 0);
        add_action('activate_facebook-messenger-live-chat-for-wordpress/facebook-messenger-live-chat-for-wordpress.php', [$this, '_activate']);
        add_action('deactivate_facebook-messenger-live-chat-for-wordpress/facebook-messenger-live-chat-for-wordpress.php', [$this, '_deactivate']);
    }

    /**
     * Do activation
     *
     * @internal  Used as a callback.
     *
     * @see  https://developer.wordpress.org/reference/functions/register_activation_hook/
     *
     * @param  bool  $network  Whether to activate this plugin on network or a single site.
     */
    public function _activate($network)
    {
        try {
            $this->preActivate();
        } catch (Exception $e) {
            if (defined('DOING_AJAX') && DOING_AJAX) {
                header('Content-Type: application/json; charset=' . get_option('blog_charset'));
                status_header(500);
                exit(json_encode([
                    'success' => false,
                    'name'    => __('Plugin Activation Error', 'facebook-messenger-live-chat-for-wordpress'),
                    'message' => $e->getMessage()
                ]));
            } else {
                exit($e->getMessage());
            }
        }

        add_option(self::OPTION_KEY, [
            'client_locale' => 'en_US',
            'chat_enabled' => 1,
            'client_page_id' => '',
            'show_on_device' => 'both',
            'show_on_page' => 'all',
            'hide_pages' => [],
            'icon_offset_left' => '',
            'icon_offset_right' => '',
            'icon_offset_bottom' => '',
            'theme_color' => '#0084FF',
            'logged_in_greeting' => '',
            'logged_out_greeting' => '',
            'greeting_dialog_delay' => 0,
        ]);
    }

    /**
     * Do installation
     *
     * @internal Used as a callback.
     */
    public function _install()
    {
        // Define useful constants.
        define('FBMLC_DIR', __DIR__ . '/');
        define('FBMLC_URI', str_replace(['http:', 'https:'], '', plugins_url('/', __FILE__)));

        // Make sure translation is available.
        load_plugin_textdomain('facebook-messenger-live-chat-for-wordpress', false, basename(__DIR__) . '/languages');

        // Load resources.
        require FBMLC_DIR . 'vendor/autoload.php';
        require FBMLC_DIR . 'includes/functions.php';
        require FBMLC_DIR . 'includes/class-messenger-chat-api.php';
        require FBMLC_DIR . 'includes/settings-page.php';
        require FBMLC_DIR . 'includes/hooks.php';
    }

    /**
     * Do deactivation
     *
     * @internal  Used as a callback.
     *
     * @see  https://developer.wordpress.org/reference/functions/register_deactivation_hook/
     *
     * @param  bool  $network  Whether to deactivate this plugin on network or a single site.
     */
    public function _deactivate($network)
    {
    }

    /**
     * Pre-activation check
     *
     * @throws  Exception
     */
    private function preActivate()
    {
        if (version_compare(PHP_VERSION, '5.6', '<')) {
            throw new Exception(__('This plugin requires PHP version 5.6 at least!', 'facebook-messenger-live-chat-for-wordpress'));
        }

        if (version_compare($GLOBALS['wp_version'], '5.2', '<')) {
            throw new Exception(__('This plugin requires WordPress version 5.2 at least!', 'facebook-messenger-live-chat-for-wordpress'));
        }
    }
}

return new Plugin(get_option(Plugin::OPTION_KEY, []));
