<?php
/**
 * Order Status Rules for WooCommerce - Section Settings
 *
 * @version 2.1.0
 * @since   1.0.0
 *
 * @author  Algoritmika Ltd.
 */

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! class_exists( 'Alg_WC_Order_Status_Rules_Settings_Section' ) ) :

class Alg_WC_Order_Status_Rules_Settings_Section {

	/**
	 * Constructor.
	 *
	 * @version 1.1.0
	 * @since   1.0.0
	 */
	function __construct() {
		add_filter( 'woocommerce_get_sections_alg_wc_order_status_rules',              array( $this, 'settings_section' ) );
		add_filter( 'woocommerce_get_settings_alg_wc_order_status_rules_' . $this->id, array( $this, 'get_settings' ), PHP_INT_MAX );
	}

	/**
	 * settings_section.
	 *
	 * @version 1.0.0
	 * @since   1.0.0
	 */
	function settings_section( $sections ) {
		$sections[ $this->id ] = $this->desc;
		return $sections;
	}

	/**
	 * get_product_title.
	 *
	 * @version 1.6.0
	 * @since   1.6.0
	 */
	function get_product_title( $product_id ) {
		return get_the_title( $product_id ) . " (#{$product_id})";
	}

	/**
	 * get_missing_product_title.
	 *
	 * @version 1.6.0
	 * @since   1.6.0
	 */
	function get_missing_product_title( $product_id ) {
		return sprintf( __( 'Product #%s', 'order-status-rules-for-woocommerce' ), $product_id );
	}

	/**
	 * get_missing_product_cat_title.
	 *
	 * @version 1.6.0
	 * @since   1.6.0
	 */
	function get_missing_product_cat_title( $term_id ) {
		return sprintf( __( 'Product category #%s', 'order-status-rules-for-woocommerce' ), $term_id );
	}

	/**
	 * get_missing_product_tag_title.
	 *
	 * @version 1.6.0
	 * @since   1.6.0
	 */
	function get_missing_product_tag_title( $term_id ) {
		return sprintf( __( 'Product tag #%s', 'order-status-rules-for-woocommerce' ), $term_id );
	}

	/**
	 * get_user_nicename.
	 *
	 * @version 1.9.0
	 * @since   1.9.0
	 */
	function get_user_nicename( $user ) {
		return sprintf( '%s (#%s)', $user->user_nicename, $user->ID );
	}

	/**
	 * get_select_all_buttons.
	 *
	 * @version 1.8.1
	 * @since   1.8.1
	 */
	function get_select_all_buttons() {
		return
			'<a href="#" class="button alg-wc-osr-select-all">'   . __( 'Select all', 'order-status-rules-for-woocommerce' )   . '</a>' . ' ' .
			'<a href="#" class="button alg-wc-osr-deselect-all">' . __( 'Deselect all', 'order-status-rules-for-woocommerce' ) . '</a>';
	}

	/**
	 * add_admin_script.
	 *
	 * @version 1.8.1
	 * @since   1.8.1
	 *
	 * @todo    [next] (dev) move this to a separate js file
	 * @todo    [next] (dev) load on needed pages only
	 */
	function add_admin_script() {
		?><script>
			jQuery( document ).ready( function() {
				jQuery( '.alg-wc-osr-select-all' ).click( function( event ) {
					event.preventDefault();
					jQuery( this ).closest( 'td' ).find( 'select.chosen_select' ).select2( 'destroy' ).find( 'option' ).prop( 'selected', 'selected' ).end().select2();
					return false;
				} );
				jQuery( '.alg-wc-osr-deselect-all' ).click( function( event ) {
					event.preventDefault();
					jQuery( this ).closest( 'td' ).find( 'select.chosen_select' ).val( '' ).change();
					return false;
				} );
			} );
		</script><?php
	}

	/**
	 * get_next_scheduled_desc.
	 *
	 * @version 1.4.0
	 * @since   1.2.0
	 *
	 * @todo    [next] (desc) `alg_wc_order_status_rules_no_history`: better title and desc
	 * @todo    [next] (desc) `crons`: desc ("... only when someone visits your site...")
	 * @todo    [maybe] (desc) Advanced Options: better description?
	 */
	function get_next_scheduled_desc() {
		if (
			'yes' === get_option( 'alg_wc_order_status_rules_use_wp_cron', 'yes' ) &&
			false != ( $next_scheduled = wp_next_scheduled( 'alg_wc_order_status_rules_process_rules', array( get_option( 'alg_wc_order_status_rules_wp_cron_schedule', 'hourly' ) ) ) )
		) {
			$date_format    = alg_wc_order_status_rules()->core->get_date_time_format();
			$current_time   = current_time( 'timestamp' );
			$server_time    = time();
			$next_scheduled = $next_scheduled + ( $current_time - $server_time );
			return sprintf( __( 'Next cron event is scheduled on %s (%s).', 'order-status-rules-for-woocommerce' ),
					'<code>' . date_i18n( $date_format, $next_scheduled ) . '</code>',
					( ( $next_scheduled - $current_time ) <= 0 ?
						__( 'i.e. now', 'order-status-rules-for-woocommerce' ) :
						sprintf( __( 'i.e. in %s', 'order-status-rules-for-woocommerce' ), human_time_diff( $next_scheduled, $current_time ) ) )
				) . ' ' .
				sprintf( __( 'Current time is %s.', 'order-status-rules-for-woocommerce' ),
					'<code>' . date_i18n( $date_format, $current_time ) . '</code>' );
		} else {
			return '';
		}
	}

	/**
	 * get_enabled_rules_desc.
	 *
	 * @version 2.0.0
	 * @since   2.0.0
	 */
	function get_enabled_rules_desc() {
		$enabled       = array_slice( get_option( 'alg_wc_order_status_rules_enabled', array() ), 0, apply_filters( 'alg_wc_order_status_rules_rules_total', 1 ), true );
		$enabled_rules = array();
		foreach ( $enabled as $rule_num => $is_enabled ) {
			if ( 'yes' === $is_enabled ) {
				$enabled_rules[] = $rule_num;
			}
		}
		return ( ! empty( $enabled_rules ) ?
			sprintf( __( 'Enabled rule(s): %s.', 'order-status-rules-for-woocommerce' ),
				__( 'Rule', 'order-status-rules-for-woocommerce' ) . ' #' . implode( ', ' . __( 'Rule', 'order-status-rules-for-woocommerce' ) . ' #', $enabled_rules ) ) :
			'' );
	}

	/**
	 * get_shipping_zones.
	 *
	 * @version 2.1.0
	 * @since   2.1.0
	 */
	function get_shipping_zones( $include_empty_zone = true ) {
		$zones = WC_Shipping_Zones::get_zones();
		if ( $include_empty_zone ) {
			$zone                                                = new WC_Shipping_Zone( 0 );
			$zones[ $zone->get_id() ]                            = $zone->get_data();
			$zones[ $zone->get_id() ]['zone_id']                 = $zone->get_id();
			$zones[ $zone->get_id() ]['formatted_zone_location'] = $zone->get_formatted_location();
			$zones[ $zone->get_id() ]['shipping_methods']        = $zone->get_shipping_methods();
		}
		return $zones;
	}

	/**
	 * get_shipping_methods_instances.
	 *
	 * @version 2.1.0
	 * @since   2.1.0
	 */
	function get_shipping_methods_instances() {
		$shipping_methods = array();
		foreach ( $this->get_shipping_zones() as $zone_id => $zone_data ) {
			foreach ( $zone_data['shipping_methods'] as $shipping_method ) {
				$shipping_methods[ $shipping_method->instance_id ] = $zone_data['zone_name'] . ': ' . $shipping_method->title;
			}
		}
		return $shipping_methods;
	}

}

endif;
