<?php namespace FBLiveChat;

use Facebook\Facebook;
use Facebook\Exceptions\FacebookSDKException;
use Facebook\Exceptions\FacebookResponseException;

/**
 * MessengerChatApi
 */
final class MessengerChatApi
{
    /**
     * Facebook connection
     *
     * @var object
     */
    private $fb;

    /**
     * Constructor
     */
    public function __construct($app_id = null, $app_secret = null)
    {
        $app_id = $app_id ? : fblivechat_get_setting('app_id');
        $app_secret = $app_secret ? : fblivechat_get_setting('app_secret');

        try {
            $this->fb = new Facebook([
                'app_id' => $app_id,
                'app_secret' => $app_secret,
                'default_graph_version' => 'v5.0',
            ]);
        } catch (FacebookResponseException $e) {
            return $e;
        } catch (FacebookSDKException $e) {
            return $e;
        }
    }

    /**
     * Validate client token
     */
    public function checkToken($token)
    {
        try {
            $extoken = $this->fb->getOAuth2Client();
            $ex_token = $extoken->debugToken($token);
            return $ex_token;
        } catch (FacebookResponseException $e) {
            return $e->getMessage();
        } catch (FacebookSDKException $e) {
            return $e->getMessage();
        }
    }

    /**
     * Get user info
     *
     * @param string $token
     */
    public function getUser($token)
    {
        try {
            $r = $this->fb->get('/me?fields=id,name,accounts,picture{url}', $token);
        } catch (FacebookResponseException $e) {
            echo esc_html__('Failed to get user info.', 'facebook-messenger-live-chat-for-wordpress') . ' ' . $e->getMessage();
            exit;
        } catch (FacebookSDKException $e) {
            echo esc_html__('Failed to get user info.', 'facebook-messenger-live-chat-for-wordpress') . ' ' . $e->getMessage();
            exit;
        }

        $user = $r->getGraphUser();

        return $user;
    }

    /**
     * Get status of token
     */
    public function getTokenStatus($token)
    {
        try {
            return $this->fb->getOAuth2Client()->debugToken($token)->getIsValid();
        } catch (FacebookResponseException $e) {
            return "error"; //$e->getMessage();
        } catch (FacebookSDKException $e) {
            return "error"; //$e->getMessage();
        }
    }

    /**
     * Get user's pages and meta info
     */
    public function getUserPages($token)
    {
        try {
            $response = $this->fb->get('/me?fields=accounts.limit(9999){picture{url},name,id,access_token}', $token); // only get picture, name, id , access_token
        } catch (FacebookResponseException $e) {
            return $e;
        } catch (FacebookSDKException $e) {
            return $e;
        }

        $pages = $response->getGraphObject()->asArray();

        return $pages;
    }

    /**
     * Get page access token with secret token
     */
    public function getPageAccessToken($token, $page_id)
    {
        try {
            $response = $this->fb->get("/$page_id?fields=access_token", $token);
        } catch (FacebookResponseException $e) {
            return $e;
        } catch (FacebookSDKException $e) {
            return $e;
        }

        $access_token = $response->getGraphObject()->asArray();

        return $access_token;
    }

    /**
     * Set App domain
     */
    public function setAppDomain($page_id, $domain, $page_access_token)
    {
        $options = [
            'setting_type' => 'domain_whitelisting',
            'whitelisted_domains' => $domain,
            'domain_action_type' => 'add',
        ];

        try {
            $r = $this->fb->post("/$page_id/thread_settings", $options, $page_access_token);
        } catch (FacebookResponseException $e) {
            return $e->getMessage();
        } catch (FacebookSDKException $e) {
            return $e->getMessage();
        }

        return $r->getGraphNode();
    }

    /**
     * Get App domain
     */
    public function getAppDomain($page_id, $page_access_token)
    {
        try {
            $r = $this->fb->get("/$page_id/thread_settings?fields=whitelisted_domains", $page_access_token);
        } catch (FacebookResponseException $e) {
            return $e->getMessage();
        } catch (FacebookSDKException $e) {
            return $e->getMessage();
        }

        return $r->getGraphList()->asArray();
    }

    /**
     * Setup webhook
     */
    public function subscribe($page_access_token, $page_id)
    {
        try {
            $r = $this->fb->post("/$page_id/subscribed_apps", [], $page_access_token);
        } catch (FacebookResponseException $e) {
            return $e->getMessage();
        } catch (FacebookSDKException $e) {
            return $e->getMessage();
        }

        return $r->getGraphNode();
    }
}
